{******************************************************************************}
{                                                                              }
{ Excel Interface Component                                                    }
{ Component ExcelUtilities.pas                                                 }
{ This interface component can be downloaded at www.tcoq.org, component page.  }
{                                                                              }
{ The contents of this file are subject to the Mozilla Public License Version  }
{ 1.1 (the "License"); you may not use this file except in compliance with the }
{ License. You may obtain a copy of the License at http://www.mozilla.org/MPL/ }
{                                                                              }
{ Software distributed under the License is distributed on an "AS IS" basis,   }
{ WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for }
{ the specific language governing rights and limitations under the License.    }
{                                                                              }
{ The Original Code is ExcelUtilities.pas                                      }
{                                                                              }
{ The Initial Developer of the Original Code is Thierry.Coq                    }
{ thierry.coq(at)centraliens.net                                               }
{ Portions created by this individual are Copyright (C)                        }
{ 2000-2008 of this individual.                                                }
{                                                                              }
{ Last modified: 2008/11/16                                                    }
{ Version 0.2                                                                  }
{                                                                              }
{******************************************************************************}
unit ExcelUtilities;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Windows, Variants, ComObj, ActiveX, Diagnostics;

// constants for excel functions and properties.
// See Excel Reference Manual for usage.
{$INCLUDE 'ExcelConstants.inc'}

const
  DISPID_ADD         =  181;
  //DISPID_ADDCOMMENT= ?? Unknown
  DISPID_BOLD        =   96;
  DISPID_CELLS       =  238;
  DISPID_COLOR       =   99;
  DISPID_COLUMN      =  240;
  DISPID_COLUMNWIDTH =  242;
  DISPID_COLUMNS     =  241;
  DISPID_COPY        =  551;
  DISPID_COUNT       =  118;
  DISPID_END         =  500;
  DISPID_FONT        =  146;
  DISPID_FORMULA     =  261;
  DISPID_HEIGHT      =  123;
  DISPID_INTERIOR    =  129;
  DISPID_ITEM        =  170;
  DISPID_QUIT        =  302;
  DISPID_REPLACE     =  226;
  DISPID_ROW         =  257;
  DISPID_ROWHEIGHT   =  272;
  DISPID_ROWS        =  258;
  DISPID_SAVE        =  283;
  DISPID_SAVEAS      = 1925;
  DISPID_SELECT      =  235;
  DISPID_SHEETS      =  485;
  DISPID_SIZE        =  104;
  DISPID_TEXT        =  138;
  DISPID_VALUE       =    6;
  DISPID_VISIBLE     =  558;
  DISPID_WIDTH       =  122;
  DISPID_WORKBOOKS   =  572;

// type
//   _Global = Excel.Application;
//     function Workbooks(Index: OleVariant): OleVariant; dispid 572;

type

  {TDispComponent is the root class to manage Dispinterfaces, in a manual manner
   waiting for the FPC compiler to catch up.}
  TDispComponent = class(TComponent)
  protected
    FDispatch: IDispatch;
    FHResult : HResult;
    FExceptInfo : TExcepInfo;
    // Error-handling routines
    Procedure CheckHResult;
    Procedure RaiseExternalException;

    // Low level routines
    Function GetIdsOfNames( aName:String): integer;
    Function BoolToVar( aValue: Boolean): TPropVariant;

    // helper procedure for calling IDispatch.Invoke.
    Procedure InvokeHelper( aDispID       : Integer;
                            aDispIDNamed  : DISPID;
                            Flags         : Word;
                            indices       : array of OleVariant;
                            aValue        : OleVariant;
                            out ArgErr    : DWord;
                            out varResult : OleVariant
                            );

    procedure PutProperty( aDispID : Integer;  anArg : TPROPVARIANT); overload;
    procedure PutProperty( aDispID : Integer;  anArg : OleVariant); overload;
    procedure PutProperty( aDispID : Integer;  indices: array of OleVariant; aValue : OleVariant); overload;
    procedure SetProperty( aName:String; aValue:Boolean); overload;
    procedure SetProperty( aPropName:String; aValue:OleVariant); overload;
    procedure SetProperty( aPropName:String; indices: array of OleVariant; aValue:OleVariant); overload;

    // GetProperty
    //=============
    //  - as a function
    function GetProperty( aName:String):OleVariant; overload;
    function GetProperty( aDispID : Integer):OleVariant; overload;

    //  - with no parameters:
    procedure GetProperty( aName:String; out aValue:IDispatch); overload;
    procedure GetProperty( aDispID : Integer; out aValue: IDispatch); overload;
    //  - with 1 parameter:
    procedure GetProperty( aName:String; index:integer; out aValue:IDispatch); overload;
    procedure GetProperty( aDispID : Integer; index:integer;out aValue: IDispatch); overload;
    //  - with 1 variant parameter:
    procedure GetProperty( aName:String; anIndex:Variant; out aValue:IDispatch); overload;
    procedure GetProperty( aDispID : Integer; anIndex:Variant; out aValue: IDispatch); overload;
    //  - with 2 parameters:
    procedure GetProperty( aName:String; i1, i2:integer; out aValue:IDispatch); overload;
    procedure GetProperty( aDispID : Integer; i1, i2:integer;out aValue: IDispatch); overload;

    // Call Procedures
    procedure CallProcedure( aName : String); overload;
    procedure CallProcedure( aDispid: Integer); overload;

    // Call procedure with a variable amount of parameters
    procedure CallProcedure( aName : String; args: array of OleVariant); overload;
    procedure CallProcedure( aDispid: Integer; args: array of OleVariant); overload;

    function CallFunctionDisp( aName : String): IDispatch; overload;
    function CallFunctionDisp( aDispid: Integer): IDispatch; overload;
    function CallFunctionDisp( aDispid: Integer; args: array of OleVariant): IDispatch; overload;

  public
    Constructor Create( anOwner : TComponent); override; overload;
    Constructor Create( anOwner : TComponent; aDispatch : IDispatch); overload;
  end;


  { TDispServer }
  { TDispServer is used to encapsulate non-Excel specific code, for the server}
  TDispServer = class( TDispComponent)
  private
    FActive: Boolean;
    FServerName: String;
    FVisible: Boolean;
    procedure SetActive(const AValue: Boolean);
    procedure SetServerName(const AValue: String);
    procedure SetVisible(const AValue: Boolean);
  protected
    procedure CreateComServer;
  public
    Constructor Create( anOwner : TComponent); override;
    Destructor Destroy; override;
    Procedure Quit;
  published
    Property Active:Boolean read FActive write SetActive;
    Property ServerName: String read FServerName write SetServerName;
    Property Visible:Boolean read FVisible write SetVisible;
  end;

  { Various Excel Objects needed for coding}
  {============================================================================}

  { TInterior }

  TInterior = class( TDispComponent)
  private
    function GetColor: Longword;
    procedure SetColor(const AValue: Longword);
  public
    // Color codes:
    // - $FF0000 is Blue
    // - $00FF00 is Green
    // - $0000FF is Red
    // - use a combination for other colors.
    property Color: Longword read GetColor write SetColor;
  end;

  { TFont }

  TFont = class( TDispComponent)
  private
    function GetBold: Boolean;
    function GetColor: Longword;
    function GetSize: Integer;
    procedure SetBold(const AValue: Boolean);
    procedure SetColor(const AValue: LongWord);
    procedure SetSize(const AValue: Integer);
  published
    // Color requires a RGB value, see the RGB function in Excel reference manual
    property Color : Longword read GetColor write SetColor;  // TODO TColor
    property Bold: Boolean read GetBold write SetBold;
    property Size: Integer read GetSize write SetSize;
  end;

  TCharacters = class(TDispComponent)
    // TODO
    // - Text
    // - Font
  end;

  TTextFrame = class(TDispComponent)
    // TODO
    // - AutoSize
    // - Characters
  end;

  TShape = class( TDispComponent)
    // TODO
    // - Visible,
    // - Width,
    // - Height,
    // - TextFrame
  end;

  TComment = class( TDispComponent)
    // TODO
    // - Text (method)
    // - Visible
    // - Shape
    // - Delete
  end;

  { TRange }

  TRange = class( TDispComponent)
  private
    function GetColumn: Long;
    function GetColumns(index: integer): TRange;
    function GetColumnWidth: Single;
    function GetFont: TFont;
    function GetFormula: String;
    function GetHeight: Long;
    function GetInterior: TInterior;
    function GetRow: Long;
    function GetRowHeight: Single;
    function GetRows(index: integer): TRange;
    function GetText: String;
    function GetValue: OleVariant;
    function GetWidth: Long;
    procedure SetColumnWidth(const AValue: Single);
    procedure SetFormula(const AValue: String);
    procedure SetRowHeight(const AValue: Single);
    procedure SetValue(const AValue: OleVariant);
    procedure SetColumnWidth(const AValue: Long);
  public
    // xlUp, or xlDown
    function End_( aDirection : LongWord) : TRange;
    function GetCount: Long;
    procedure AddComment( aText:String);
    procedure Copy;
    procedure Replace( What,
                       Replacement: String;
                       Lookat: long;
                       SearchOrder: long;
                       MatchCase: Boolean);

    procedure Select;
    property Rows[index:integer] : TRange read GetRows;
    property Columns[index:integer] : TRange read GetColumns;
  published
    property Column: Long read GetColumn;
    property ColumnWidth: Single read GetColumnWidth write SetColumnWidth;
    property Count: Long read GetCount;
    property Font: TFont read GetFont;
    property Formula: String read GetFormula write SetFormula;
    property Height: Long read GetHeight;
    property Interior: TInterior read GetInterior;
    property Row: Long read GetRow;
    property RowHeight: Single read GetRowHeight write SetRowHeight;
    property Text: String read GetText;
    property Value: OleVariant read GetValue write SetValue;
    property Width: Long read GetWidth;
    //TODO: additional properties
    //Format properties
    //  - BorderAround
    //  - Borders
    //  - WrapText
    //  - VerticalAlignement
    //  - HorizontalAlignment
    //  - NumberFormat
    //  - Comment
  end;

  { TSheet }

  TSheet = class( TDispComponent)
    function Cells(RowIndex, ColumnIndex:Integer): TRange;
    function Rows: TRange;
    //TODO: fonctions additionnelles
    //Let DerniereDepense = FeuilleDepenses.Cells(Rows.Count, 3).End(xlUp).Row
    //  - Rows
    //  - Paste( (Destination))

  end;

  { TSheets }

  TSheets = class( TDispComponent)
  public
    function Sheet(index: OleVariant): TSheet;
  end;

  { TWorkBook }

  TWorkBook = class( TDispComponent)
  public
    procedure Save;
    procedure SaveAs( aFileName : String);
    function Sheets : TSheets;
  end;

  { TWorkBooks }

  TWorkBooks = class( TDispComponent)
  public
    function Add : TWorkBook;
    function Open( aFileName: String) : TWorkBook;
  end;

  { TExcelApplication }
  TExcelApplication = class(TDispServer)
  private
    FWorkBooks: TWorkBooks;
    function GetWorkBooks: TWorkBooks;
  public
    Constructor Create( anOwner : TComponent); override;
    Destructor Destroy; override;
  published
    property WorkBooks: TWorkBooks read GetWorkBooks;
    //TODO: properties to add
    // - Selection : TRange
  end;



implementation

{ TExcelApplication }


function TExcelApplication.GetWorkBooks: TWorkBooks;
var
  aDisp : IDispatch;
begin
  if not assigned( FWorkBooks) then
  begin
    //get the property,
    //return the IDispatch
    //create the Workbook and return it
    GetProperty( DISPID_WORKBOOKS, aDisp);
    FWorkBooks := TWorkBooks.Create( self, aDisp);
  end;
  result := FWorkBooks;
end;

constructor TExcelApplication.Create( anOwner:TComponent);
begin
  inherited Create(anOwner);
  ServerName := 'Excel.Application';
end;

destructor TExcelApplication.Destroy;
begin
  inherited Destroy;
end;

{ TDispServer }

procedure TDispServer.SetServerName(const AValue: String);
begin
  if FServerName=AValue then exit;
  FServerName:=AValue;
end;

procedure TDispServer.SetVisible(const AValue: Boolean);
begin
  if FVisible=AValue then exit;
  PutProperty( DISPID_VISIBLE, aValue);
  FVisible:=AValue;
end;

procedure TDispServer.CreateComServer;
var
  GuID : TGUID;
begin
  //Preconditions:
  If FDispatch <> nil then exit;

  FDispatch := nil;
  GuID := ProgIDToClassID(ServerName);
  FHResult := CoCreateInstance(GuiD,nil,CLSCTX_INPROC_SERVER or CLSCTX_LOCAL_SERVER, IDispatch, FDispatch);
  CheckHResult;
end;

procedure TDispServer.SetActive(const AValue: Boolean);
begin
  if FActive=AValue then exit;

  if AValue then
    CreateComServer
  else
    Quit;

  FActive:=AValue;
end;

constructor TDispServer.Create(anOwner: TComponent);
begin
  inherited Create(anOwner);
end;

destructor TDispServer.Destroy;
begin
  if Active then Quit;
  inherited Destroy;
end;

procedure TDispServer.Quit;
begin
  if Not Active then Exit;
  FActive := false;
  CallProcedure( DISPID_QUIT);
end;

{ TDispComponent }

procedure TDispComponent.CheckHResult;
begin
  If FHResult <> 0 then
  begin
    Case FHResult of
      DISP_E_EXCEPTION : RaiseExternalException;
    else
      Raise Exception.Create( 'HResult = '+ IntToHex(FHResult,8));
    end;
  end;
end;

procedure TDispComponent.RaiseExternalException;
begin
  Raise Exception.Create( 'External software raised an exception, '
    + 'HResult = "'+ IntToHex(FHResult,8)+'", '
    + 'Exception SCode = "'+IntToHex(FExceptInfo.Scode,8)+'"'
    );
end;

function TDispComponent.GetIdsOfNames(aName: String): integer;
var
  lDispId: Integer;
  member : WideString;
begin
  member := aName;
  FHResult := FDispatch.GetIDsOfNames(GUID_NULL, @member, 1, LOCALE_USER_DEFAULT, @lDispID);
  Result := lDispId;
  logMessage('GetIDsOfNames on "'+aName+'", DispID="'+IntToStr(Result)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;
end;

function TDispComponent.BoolToVar(aValue: Boolean): TPropVariant;
begin
  Result.VT := VT_BOOL;
  Result.boolVal:= aValue;
end;

procedure TDispComponent.InvokeHelper(aDispID: Integer; aDispIDNamed: DISPID;
  Flags: Word; indices: array of OleVariant; aValue: OleVariant;
  out ArgErr: DWord; out varResult: OleVariant);
var
  Params: TDispParams;
  LocalVariantArg : array of OleVariant;
  dispidNamed :  DISPID;
  iParam : integer;
  ParamNumber : Integer;
  IndicesNumber: Integer;
  ParamIncrement: Integer;
begin
  // Initialization
  dispidNamed := aDispIDNamed;
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}

  // Setting up parameter list
  ParamIncrement:=0;
  IndicesNumber := Length(indices);
  ParamNumber := IndicesNumber;
  if aDispIDNamed=DISPID_PROPERTYPUT then       // add one for the property_put
    inc(ParamIncrement);                        // value.
  ParamNumber := IndicesNumber + ParamIncrement;

  if ParamNumber>0 then
  begin
    SetLength( LocalVariantArg, ParamNumber);
    for iParam := 0 to IndicesNumber-1 do
      LocalVariantarg[ ParamNumber - iParam - (1-ParamIncrement)]
        := indices[iParam];
    if aDispIDNamed=DISPID_PROPERTYPUT then     // property_put Value
      LocalVariantArg[0] := aValue;
    Params.cArgs := ParamNumber;
    Params.rgvarg := @LocalVariantArg[0];
  end;

  // setting up named args (0 or 1 only is managed)
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs := nil;
  if aDispIDNamed <> 0 then
  begin
    Params.cNamedArgs := 1;
    Params.rgdispidNamedArgs:= @dispidNamed;
  end;

  // Calling invoke now
  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             Flags, Params,
                             @varResult, @FExceptInfo, @ArgErr);

  logMessage(
    'InvokeHelper DispID="'+IntToStr(aDispID) +'", '
   +'HResult =' +IntToHex(FHResult, 12)+', '
    +'aValue = "' + VarToSTr(aValue)+'"'
   );
  CheckHResult;

end;



procedure TDispComponent.PutProperty( aDispID : Integer; anArg : TPROPVARIANT);
var
  Params: TDispParams;
  ArgErr : DWord;
  LocalVariantArg : array[0..0] of TPROPVARIANT;
  dispidNamed :  DISPID;
begin
  // Initialization
  dispidNamed := DISPID_PROPERTYPUT;
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  FillChar(LocalVariantArg,SizeOf(LocalVariantArg),0);
  {$HINTS ON}

  LocalVariantArg[0] := anArg;
  Params.cArgs := 1;
  Params.cNamedArgs := 1;
  Params.rgdispidNamedArgs:= @dispidNamed;
  Params.rgvarg := @LocalVariantArg[0];

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYPUT, Params,
                             nil, @FExceptInfo, @ArgErr);

  logMessage('PutProperty DispID="'+IntToStr(aDispID)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;

end;

procedure TDispComponent.PutProperty(aDispID: Integer; anArg: OleVariant);
var
  Params: TDispParams;
  ArgErr : DWord;
  LocalVariantArg : OleVariant;
  dispidNamed :  DISPID;
begin
  // Initialization
  dispidNamed := DISPID_PROPERTYPUT;
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  FillChar(LocalVariantArg,SizeOf(LocalVariantArg),0);
  {$HINTS ON}

  LocalVariantArg := anArg;
  Params.cArgs := 1;
  Params.cNamedArgs := 1;
  Params.rgdispidNamedArgs:= @dispidNamed;
  Params.rgvarg := @LocalVariantArg;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYPUT, Params,
                             nil, @FExceptInfo, @ArgErr);

  logMessage('PutProperty-Variant DispID="'+IntToStr(aDispID)
    +'", HResult =' +IntToStr(FHResult)+', '
    +'anArg = "' + VarToSTr(anArg) + '"'
    );
  CheckHResult;

end;

procedure TDispComponent.PutProperty(aDispID: Integer;
  indices: array of OleVariant; aValue: OleVariant);
var
  varResult: OleVariant;
  ArgErr : DWord;
begin
  InvokeHelper( aDispid, DISPID_PROPERTYPUT, DISPATCH_PROPERTYPUT, indices,
                aValue, ArgErr, varResult);
end;

procedure TDispComponent.SetProperty(aName: String; aValue: Boolean);
var
  propDispId : Integer;
  propArg : TPROPVARIANT;
begin
  propDispId := GetIdsOfNames( aName);
  propArg := BoolToVar( aValue);
  PutProperty( propDispId, propArg);
end;

procedure TDispComponent.SetProperty(aPropName: String; aValue: OleVariant);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aPropName);
  PutProperty( propDispId, aValue);
end;

procedure TDispComponent.SetProperty(aPropName: String;
  indices: array of OleVariant; aValue: OleVariant);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aPropName);
  PutProperty( propDispId, indices, aValue);
end;

function TDispComponent.GetProperty(aName: String): OleVariant;
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  result := GetProperty(propDispId);
end;

function TDispComponent.GetProperty(aDispID: Integer): OleVariant;
var
  Params: TDispParams;
  ArgErr : DWord;
  vrResult : OleVariant;

begin

  // Initialization
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}

  Params.cArgs := 0;
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs:= nil;
  Params.rgvarg := nil;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYGET, Params,
                             @vrResult, @FExceptInfo, @ArgErr);
  logMessage('Function GetProperty DispID="'+IntToStr(aDispID)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;

  result := vrResult;
end;


procedure TDispComponent.GetProperty(aName: String; out aValue: IDispatch);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  GetProperty( propDispId, aValue);
end;

procedure TDispComponent.GetProperty( aDispID: Integer;
                                      out aValue: IDispatch);
var
  Params: TDispParams;
  ArgErr : DWord;
  vrResult : TPropVariant;
begin
  aValue := nil;

  // Initialization
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}

  Params.cArgs := 0;
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs:= nil;
  Params.rgvarg := nil;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYGET, Params,
                             @vrResult, @FExceptInfo, @ArgErr);
  logMessage('GetProperty DispID="'+IntToStr(aDispID)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;

  case vrResult.vt of
    VT_EMPTY, VT_NULL:
      begin {do nothing} end;
    VT_DISPATCH:
      aValue := IDispatch(vrResult.pdispVal);
    else
      raise Exception.Create( 'unknown and unhandled result from GetProperty')
  end;


end;

procedure TDispComponent.GetProperty(aName: String; index: integer;
  out aValue: IDispatch);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  GetProperty( propDispId, index, aValue);
end;

procedure TDispComponent.GetProperty(aDispID: Integer; index: integer;
  out aValue: IDispatch);
var
  Params: TDispParams;
  ArgErr : DWord;
  vrResult : TPropVariant;
  localVarArray: TPropVariant;
begin
  aValue := nil;

  // Initialization
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}


  localVarArray.vt := VT_INT;
  localVarArray.intVal := index;
  Params.cArgs := 1;
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs:= nil;
  Params.rgvarg := @localVarArray;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYGET, Params,
                             @vrResult, @FExceptInfo, @ArgErr);
  logMessage( 'GetPropertyIndex, DispID="'+IntToStr(aDispID)
    +'", Index = ' + IntToStr(Index)
    +', HResult =' +IntToStr(FHResult));
  CheckHResult;

  case vrResult.vt of
    VT_EMPTY, VT_NULL:
      begin {do nothing} end;
    VT_DISPATCH:
      aValue := IDispatch(vrResult.pdispVal);
    else
      raise Exception.Create( 'unknown and unhandled result from GetProperty')
  end;
end;

procedure TDispComponent.GetProperty(aName: String; anIndex: Variant;
  out aValue: IDispatch);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  GetProperty( propDispId, anIndex, aValue);
end;

{ defined outside procedure for long life}
procedure TDispComponent.GetProperty(aDispID: Integer; anIndex: Variant;
  out aValue: IDispatch);
var
  Params: TDispParams;
  vrResult : TPropVariant;
  localVarArray: OleVariant;
begin
  aValue := nil;
  {$HINTS OFF}
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}

  localVarArray := anIndex;
  Params.cArgs := 1;
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs:= nil;
  Params.rgvarg := @localVarArray;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYGET, Params,
                             @vrResult, nil, nil);
  logMessage( 'GetPropertyIndex, DispID="'+IntToStr(aDispID)
    +'", Index = ' + anIndex
    +', HResult =' +IntToHex(FHResult, 16)
    );
  CheckHResult;

  case vrResult.vt of
    VT_EMPTY, VT_NULL:
      begin {do nothing} end;
    VT_DISPATCH:
      aValue := IDispatch(vrResult.pdispVal);
    else
      raise Exception.Create( 'unknown and unhandled result from GetProperty')
  end;
end;

procedure TDispComponent.GetProperty(aName: String; i1, i2: integer;
  out aValue: IDispatch);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  GetProperty( propDispId, i1, i2, aValue);
end;

procedure TDispComponent.GetProperty(aDispID: Integer; i1, i2: integer;
  out aValue: IDispatch);
var
  Params: TDispParams;
  ArgErr : DWord;
  vrResult : TPropVariant;
  localVarArray: array[0..1] of OleVariant;
begin
  aValue := nil;

  // Initialization
  {$HINTS OFF}
  FillChar(FExceptInfo, SizeOf(ExcepInfo),0);
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}

  // arguments are inversed
  //localVarArray[0].vt := VT_I2;
  //localVarArray[0].intVal := i2;
  //localVarArray[1].vt := VT_I2;
  //localVarArray[1].intVal := i1;
  localVarArray[0] := i2;
  localVarArray[1] := i1;
  Params.cArgs := 2;
  Params.cNamedArgs := 0;
  Params.rgdispidNamedArgs:= nil;
  Params.rgvarg := @localVarArray;

  FHResult := FDispatch.Invoke( aDispID, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                             DISPATCH_PROPERTYGET, Params,
                             @vrResult, @FExceptInfo, @ArgErr);

  // We get a SIGSEV error here: access violation on VarR4FromCy incorrectly called.
  logMessage( 'GetPropertyIndex, DispID="'+IntToStr(aDispID)
    +'", (I1, I2) = (' + IntToStr(i1)+', '+ IntToStr(i2)+')'
    +', HResult =' +IntToStr(FHResult));
  CheckHResult;

  case vrResult.vt of
    VT_EMPTY, VT_NULL:
      begin {do nothing} end;
    VT_DISPATCH:
      aValue := IDispatch(vrResult.pdispVal);
    else
      raise Exception.Create( 'unknown and unhandled result from GetProperty')
  end;
end;

procedure TDispComponent.CallProcedure(aName: String);
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  CallProcedure( propDispId);
end;

procedure TDispComponent.CallProcedure(aDispid: Integer);
var
  Params: TDispParams;
begin
  {$HINTS OFF}
  FillChar(Params, SizeOf(DispParams),0);
  {$HINTS ON}
  //Params.cArgs             := 0;
  //Params.cNamedArgs        := 0;
  //Params.rgvarg            := nil;
  //Params.rgdispidNamedArgs := nil;
  FHResult := FDispatch.Invoke( aDispid, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                                DISPATCH_METHOD, Params,
                                nil, nil, nil);
  logMessage('CallProcedure DispID="'+IntToStr(aDispID)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;
end;

procedure TDispComponent.CallProcedure(aName: String; args: array of OleVariant
  );
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  CallProcedure( propDispId, args);
end;

procedure TDispComponent.CallProcedure(aDispid: Integer;
  args: array of OleVariant);
var
  varResult: OleVariant;
  ArgErr : DWord;
begin
  InvokeHelper( aDispid, 0, DISPATCH_METHOD, args,
                Null, ArgErr, varResult);
end;

function TDispComponent.CallFunctionDisp(aName: String): IDispatch;
var
  propDispId : Integer;
begin
  propDispId := GetIdsOfNames( aName);
  result := CallFunctionDisp( propDispId);
end;

function TDispComponent.CallFunctionDisp(aDispid: Integer): IDispatch;
var
  Params: TDispParams;
  vrResult : TPropVariant;
begin
  Params.cArgs             := 0;
  Params.cNamedArgs        := 0;
  Params.rgvarg            := nil;
  Params.rgdispidNamedArgs := nil;
  FHResult := FDispatch.Invoke( aDispid, GUID_NULL, LOCALE_SYSTEM_DEFAULT,
                                DISPATCH_METHOD, Params,
                                @vrResult, nil, nil);
  logMessage('CallFunctionDisp DispID="'+IntToStr(aDispID)+'", HResult =' +IntToStr(FHResult));
  CheckHResult;

  case vrResult.vt of
    VT_EMPTY, VT_NULL:
      begin {do nothing} end;
    VT_DISPATCH:
      result := IDispatch(vrResult.pdispVal);
    else
      raise Exception.Create( 'unknown and unhandled result from GetProperty')
  end;
end;

function TDispComponent.CallFunctionDisp(aDispid: Integer;
  args: array of OleVariant): IDispatch;
var
  varResult: OleVariant;
  ArgErr : DWord;
begin
  result := nil;
  InvokeHelper( aDispid, 0, DISPATCH_METHOD, args,
                null, ArgErr, varResult);
  result := varResult;
end;

constructor TDispComponent.Create(anOwner: TComponent);
begin
  inherited Create(anOwner);
end;

constructor TDispComponent.Create(anOwner: TComponent; aDispatch: IDispatch);
begin
  inherited Create( anOwner);
  FDispatch := aDispatch;
end;

{ TWorkBooks }

function TWorkBooks.Add: TWorkBook;
var
  aVar :IDispatch;
begin
  aVar := CallFunctionDisp( DISPID_ADD);
  result := TWorkBook.Create(self, aVar) ;
end;

function TWorkBooks.Open(aFileName: String): TWorkBook;
var
  aVar :IDispatch;
begin
  aVar := CallFunctionDisp( DISPID_ADD, [aFileName]);
  result := TWorkBook.Create(self, aVar) ;
end;

{ TWorkBook }

procedure TWorkBook.Save;
begin
  CallProcedure(DISPID_SAVE);
end;

procedure TWorkBook.SaveAs(aFileName: String);
var
  varFileName : OleVariant;
begin
  varFileName := aFileName;
  CallProcedure( DISPID_SAVEAS, [varFileName]);
end;

function TWorkBook.Sheets: TSheets;
var
  aVar :IDispatch;
begin
  //aVar := CallFunctionDisp( 'Sheets');
  GetProperty(DISPID_SHEETS, aVar);
  result := TSheets.Create(self, aVar) ;
end;

{ TSheets }
function TSheets.Sheet(index: OleVariant): TSheet;
var
  aVar :IDispatch;
begin
  GetProperty(DISPID_ITEM, index, aVar);
  result := TSheet.Create(self, aVar);
end;

{ TSheet }

function TSheet.Cells(RowIndex, ColumnIndex: Integer): TRange;
var
  aVar :IDispatch;
begin
  GetProperty(DISPID_CELLS, RowIndex, ColumnIndex, aVar);
  result := TRange.Create(self, aVar);
end;

function TSheet.Rows: TRange;
var
  aDisp : IDispatch;
begin
  GetProperty(DISPID_ROWS,  aDisp);
  result := TRange.Create( self, aDisp);
end;

{ TRange }

function TRange.GetColumn: Long;
begin
  result := GetProperty( DISPID_COLUMN);
end;

function TRange.GetColumns(index: integer): TRange;
var
  aDisp : IDispatch;
begin
  GetProperty(DISPID_COLUMNS, index, aDisp);
  result := TRange.Create( self, aDisp);
end;

function TRange.GetColumnWidth: Single;
begin
  result := GetProperty( DISPID_COLUMNWIDTH);
end;

function TRange.GetFont: TFont;
var
  aDisp : IDispatch;
begin
  GetProperty(DISPID_FONT, aDisp);
  result := TFont.Create( self, aDisp);
end;

function TRange.GetFormula: String;
begin
  result := GetProperty( DISPID_FORMULA);
end;

function TRange.GetHeight: Long;
begin
  result := GetProperty( DISPID_HEIGHT);
end;

function TRange.GetInterior: TInterior;
var
  aDisp : IDispatch;
begin
  GetProperty(DISPID_INTERIOR, aDisp);
  result := TInterior.Create( self, aDisp);
end;

function TRange.GetRow: Long;
begin
  result := GetProperty( DISPID_ROW);
end;

function TRange.GetRowHeight: Single;
begin
  result := GetProperty(DISPID_ROWHEIGHT);
end;

function TRange.GetRows(index: integer): TRange;
var
  aDisp : IDispatch;
begin
  GetProperty(DISPID_ROWS, index, aDisp);
  result := TRange.Create( self, aDisp);
end;

function TRange.GetText: String;
begin
  result := GetProperty(DISPID_TEXT);
end;

function TRange.GetValue: OleVariant;
begin
  result := GetProperty(DISPID_VALUE);
end;

function TRange.GetWidth: Long;
begin
  result := GetProperty( DISPID_WIDTH);
end;

procedure TRange.SetColumnWidth(const AValue: Single);
begin
  PutProperty( DISPID_COLUMNWIDTH, AValue);
end;

procedure TRange.SetFormula(const AValue: String);
begin
  PutProperty(DISPID_FORMULA, [], AValue);
end;

procedure TRange.SetRowHeight(const AValue: Single);
begin
  PutProperty( DISPID_ROWHEIGHT, AValue);
end;

procedure TRange.SetValue(const AValue: OleVariant);
begin
  //SetProperty('Value', AValue);
  PutProperty(DISPID_VALUE, [], AValue);
end;

procedure TRange.SetColumnWidth(const AValue: Long);
begin
  PutProperty( DISPID_COLUMNWIDTH, AValue);
end;

function TRange.End_(aDirection: LongWord): TRange;
var
  aVar : IDispatch;
begin
  GetProperty( DISPID_END, aDirection, aVar);
  result := TRange.Create(Self.Owner, aVar);
end;

function TRange.GetCount: Long;
begin
  result := GetProperty( DISPID_COUNT);
end;

procedure TRange.AddComment(aText: String);
begin
  CallProcedure('AddComment', [aText] );
end;

procedure TRange.Select;
begin
  CallProcedure( DISPID_SELECT);
end;

procedure TRange.Copy;
begin
  CallProcedure( DISPID_COPY);
end;

procedure TRange.Replace(What, Replacement: String; Lookat: long;
  SearchOrder: long; MatchCase: Boolean);
begin
  CallProcedure(DISPID_REPLACE, [what, replacement, lookat, searchorder, matchcase]);
end;

{ TFont }

function TFont.GetBold: Boolean;
begin
  result := GetProperty(DISPID_BOLD);
end;

function TFont.GetColor: Longword;
begin
  result := GetProperty(DISPID_COLOR);
end;

function TFont.GetSize: Integer;
begin
  result := GetProperty(DISPID_SIZE);
end;


procedure TFont.SetBold(const AValue: Boolean);
begin
  PutProperty( DISPID_BOLD, [], AValue);
end;

procedure TFont.SetColor(const AValue: LongWord);
begin
  PutProperty(DISPID_COLOR, [], AValue);
end;

procedure TFont.SetSize(const AValue: Integer);
begin
  PutProperty( DISPID_SIZE, [], AValue);
end;

{ TInterior }

function TInterior.GetColor: Longword;
begin
  result := GetProperty(DISPID_COLOR);
end;

procedure TInterior.SetColor(const AValue: Longword);
begin
  PutProperty(DISPID_COLOR, [], AValue);
end;

end.

